; Helper scripts for the Teamtalk application.
;
; Author:  Doug Lee

/*
Design note:
These scripts use the Scripting.Dictionary object available in all
supported versions of Windows, rather than the Collection data type
included in the JAWS scripting language starting in JAWS 11 update 2,
so that they can compile and run under older JAWS versions. The usage
and syntax of Dictionary objects, compared to Collections, is a bit
slower and messier but is not problematic unless hundreds of
references to a Dictionary object occur in one second. Would-be
modifiers of these scripts should take care to minimize such
references in loops.
[DGL, 2011-12-18]
*/


; This makes string comparisons work as in JAWS 12 and older,
; making "RichEdit" match "RichEdit20W" for example.
;#pragma StringComparison partial


include "hjconst.jsh"
include "tt4_jcpdict.jsm"

;========================== Module jcp (from jcpdict) ==========================
; Control property manager for JAWS using the Scripting.Dictionary object.
; Callback: jcp__setCustomProps.
; Set all control-specific properties there to make them work in speech and Braille consistently.
;
; Author:  Doug Lee

include "winstyles.jsh"
const
; Default cache lifetime for jcp__setAll() in milliseconds.
	jcp___DefaultCacheTicks = 500

globals
; Master on/off switch for the system, set by jcp__enable(), used for testing.
	int jcp___disabled,
; Cache lifetime in effect at any given time.
	int jcp___cacheTicks,
; True when this system is already active (used to avoid "unknown function call" messages)
	int jcp___inside,
; Last script key and name of script that was just run.
; Set by KeyPressedEvent (via jcp___setCaller) and used by jcp__getActionClass().
	string jcp___key,
	string jcp___caller,
; Caches for getObjectSubtypeCode() and jcp__getID(),
; because they can get called below enough times per second to slow things down.
; Example symptom: A one-second delay on a PCCursor() script call.
	object jcp___intcache,
	int jcp___intTick

variant function jcp___vcast(variant v)
return v
endFunction

void function autoStartEvent()
let jcp___cacheTicks = jcp___DefaultCacheTicks
let jcp___intcache = createObjectEx("Scripting.Dictionary", False)
let jcp___intcache.CompareMode = 1  ; TextCompare
endFunction

int function jcp__enable(int flag)
; Master on/off switch:  pass 0 to turn off, 1 for on, 2 to query without changing, and anything else to toggle.
; (This arrangement was chosen to make direct on/off make sense while making it easy to use isSameScript() to query on one press and toggle on more.)
if flag == 0 || flag == 1 then
	let jcp___disabled = !flag
elif flag != 2 then
	let jcp___disabled = !jcp___disabled
endIf
if jcp___disabled then
	sayMessage(OT_No_Disable, "JCP disabled")
else
	sayMessage(OT_No_Disable, "JCP enabled")
endIf
return !jcp___disabled
endFunction

int function jcp__cache(int ticks)
; Set how long the cache of control properties can be kept before a forced refresh.
; Determines how often setCustomControlProps() is called.
; Pass 0 to use the internal default value.
; WARNING: Setting this too low can cause a major performance hit on systems with Braille displays,
; because setCustomControlProps() can be called over 40 times per second in such cases.
; Returns the old value that is being replaced.
; ticks: The maximum number of ticks (milliseconds) to keep the cache valid,
; or 0 to use the internal default value.
var
	int old
let old = jcp___cacheTicks
if !ticks then
	let ticks = jcp___DefaultCacheTicks
endIf
let jcp___cacheTicks = ticks
return old
endFunction

int function jcp__isEmpty(variant props)
; Returns True if props represents an empty property set.
return stringToInt(props.count) == 0
endFunction

variant function jcp__new()
; Returns an empty props structure.
return createObjectEx("Scripting.Dictionary", False)
endFunction

string function jcp__getActionClass()
; Get the type of action that appears to have caused the current jcpDict invocations:
;	tab:  Navigation among controls.
;	arrow:  Navigation within a control.
var
	string id
let id = stringLower(jcp___key)
if stringContains(id, "tab") then
	return "tab"
elif stringContains(id, "arrow")
|| stringContains(id, "home")
|| stringContains(id, "end") then
	return "arrow"
endIf
return ""
endFunction

variant function jcp__get(object props, string whichProp)
; Get the value of whichProp from props and return it.
; Case is not significant in property names.
; Dotted subclassing like fieldname.brl is permitted:
; If fieldname.brl exists it is returned, else if fieldname exists it is returned, else null is returned.
; Implemented subclassings:
;	- .brl for Braille calls.
;	- .<caller> for functions in this file that call jcp__get().
;	- .<class> for keys/scripts that initiated this jcp__get call (see jcp__getActionClass()).
; The latter of these is implemented internally here, not passed in whichProp.
; <caller> takes precedence over <class>.
; If a value begins with @, it is taken as a function call, and the return value of the call is returned.
var
	string wp,
	int pos,
	string val,
	string class,
	int done
let wp = stringLower(whichProp)
let val = ""
let done = False
let class = ""
while !done
	let val = ""
	if props.exists(wp) then
		; The .exists test keeps the wp key from springing into existance and skewing props.count.  [DGL, 2007-11-23]
		let val = props(wp)
	endIf
	if val then
		if val == "@" then
			let val = formatStringWithEmbeddedFunctions("<" +stringChopLeft(val, 1) +">")
		endIf
		return val
	endIf
	let pos = stringContainsFromRight(wp, ".")
	if pos then
		let wp = stringLeft(wp, pos-1)
		if !wp then
			let done = True
		endIf
	elif !class then
		let class = jcp__getActionClass()
		if class then
			let wp = wp +"." +jcp__getActionClass()
		else
			let done = True
		endIf
	else
		let done = True
	endIf
endWhile
return ""
endFunction

int function jcp__prepend(object byRef props, string whichProp, string s)
; Prepend s to the property value indicated.
; whichProp should indicate a string property, not a numeric one.
var string val let val = jcp__get(props, whichProp)
if val then
	let val = s +" " +val
else
	let val = s
endIf
jcp__set(props, whichProp, val)
endFunction

int function jcp__append(object byRef props, string whichProp, string s)
; Append s to the property value indicated.
; whichProp should indicate a string property, not a numeric one.
var string val let val = jcp__get(props, whichProp)
if val then
	let val = val +" " +s
else
	let val = s
endIf
jcp__set(props, whichProp, val)
endFunction

void function jcp__setBits(object byRef props, string whichProp, int bits)
; Logically OR a bit or set of bits into a property value.
; whichProp should indicate a numeric property, not a string one.
var int val let val = jcp__get(props, whichProp) +0
let val = val | bits
jcp__set(props, whichProp, val)
endFunction

void function jcp__clearBits(object byRef props, string whichProp, int bits)
; Logically AND a bit or set of bits out of a property value.
; whichProp should indicate a numeric property, not a string one.
var int val let val = jcp__get(props, whichProp) +0
let val = val & (~bits)
jcp__set(props, whichProp, val)
endFunction

variant function jcp__setItemPos(variant byRef props, variant pos, variant cnt)
; Set the Position property by its component parts.
; Also sets ItemPos and ItemCount.
; Recommended over setting Position or ItemPos/ItemCount directly.
; For convenience, stringToInt conversions are performed here if necessary
; so callers don't have to do it.
var int p let p = stringToInt(pos)
var int c let c = stringToInt(cnt)
jcp__set(props, "ItemPos", p)
jcp__set(props, "ItemCount", c)
var string buf
if p then
	let buf = formatString(jcp___MOfN, intToString(p), intToString(c))
else
	let buf = formatString(jcp___NItems, intToString(c))
endIf
jcp__set(props, "Position", buf)
return True
endFunction

int function jcp__setTablePos(variant byRef props, int rowIdx, int rowCount, int colIdx, int colCount)
; Set table row/column position and count properties.
; The "+0"'s avoid beeps when nulls are passed in.
return jcp__set(props, "row", rowIdx+0)
&& jcp__set(props, "rows", rowCount+0)
&& jcp__set(props, "col", colIdx+0)
&& jcp__set(props, "cols", colCount+0)
endFunction

string function jcp__getID(string sOrigin)
var
	int left, int right, int top, int bottom
if sOrigin == "current" || (sOrigin == "focus" && isPCCursor()) then
	if getObjectRect(left, right, top, bottom) then
		return "obj" +intToString(left) +"," +intToString(top)
	else
		; Use the active cursor location.
		let left = getCursorCol()
		let top = getCursorRow()
		return "point" +intToString(left) +"," +intToString(top)
	endIf
elif sOrigin == "focus" then
	; Focus but PC cursor not active; use the PC cursor location explicitly.
	saveCursor()  PCCursor()
	let left = getCursorCol()
	let top = getCursorRow()
	restoreCursor()
	return "point" +intToString(left) +"," +intToString(top)
else
	; sOrigin is not focus or current.  Use sOrigin itself as an ID.
	return sOrigin
endIf
endFunction

globals
	int jcp___lastTick,
	string jcp___lastID,
	string jcp___lastProps
int function jcp__setAll(variant byRef props, string sOrigin, /*?*/ int forceNoCache)
; Sets props to be a structure for other controlprop* functions.
; Obtains any properties of the indicated control that stray from what JAWS would normally discover and report.
; Returns True if anything was done or null if no special handling of this control is necessary.
var
	string id,
	int tc
if userBufferIsActive() then
	return False
endIf
if !sOrigin then
	let sOrigin = "focus"
endIf

; Cache property sets (avoids serious CPU hits when a Braille display is active)
let tc = getTickCount()
let id = jcp___cachedVal(sOrigin)
if !forceNoCache && tc -jcp___lastTick <= jcp___cacheTicks
&& id && !stringCompare(jcp___lastID, id, True) then
	let props = jcp___vcast(jcp___lastProps)
	return !jcp__isEmpty(props)
endIf
let jcp___lastID = jcp___vcast(id)
let jcp___lastTick = tc

; Start with a clean slate.
let props = jcp__new()
let jcp___lastProps = jcp___vcast(props)

; Set any custom properties via the callback function.
;var int tcj let tcj = getTickCount()
jcp__setCustomProps(props, sOrigin)
;sayString(formatString("%2", id, intToString(getTickCount()-tcj)))
if !jcp__isEmpty(props) then
	; Set the origin so jcp__say() will know where to get defaults.
	if !stringLength(jcp__get(props, "origin")) then
		jcp__set(props, "origin", sOrigin)
	endIf
	let jcp___lastProps = jcp___vcast(props)
	return True
endIf
return False
endFunction

int function jcp__setByOrigin(string sOrigin, string what, handle byRef hwnd, object byRef o, int byRef childID)
; Set origin info by reference.
; Allowed values for What:
;	h - Window handle.
;	m - MSAA object and childID
;	n - Nav ID (implies h).
;	c - Cursor-centric function support (e.g., getObjectName etc.).
; If passing c, first activate the cursor that will be used for getObject* etc. calls.
; At present, m and n are mutually exclusive.
; Returns True on success and False on failure.
let what = stringLower(what)

; hwnd origin case.
if sOrigin == "hwnd" then
	let hwnd = stringToHandle(stringChopLeft(sOrigin, 4))
	; Hwnd comes back whether requested or not here.
	if stringContains(what, "m") then
		let o = getObjectFromEvent(hwnd, -4, 0, childID)
	elif stringContains(what, "n") then
		; ToDo: Probably insufficient.
		let childID = jcp___vcast(navGetCurrentObjectID())
	endIf
	if stringContains(what, "c") then
		if getCurrentWindow() != hwnd then
			moveToWindow(hwnd)
			; TODO:  Verify that this returns the right window.
		endIf
	endIf
	return True
endIf

; Focus or current-when-current-is-focus cases.
if sOrigin == "focus" || (sOrigin == "current" && isPCCursor()) then
	let hwnd = getFocus()
	if stringContains(what, "m") then
		let o = getFocusObject(childID)
		if !o then
			return False
		endIf
	elif stringContains(what, "n") then
		let childID = jcp___vcast(navGetFocusObjectID())
	endIf
	if !isPCCursor() then
		; Make GetObject*() calls apply to the right thing.
		saveCursor()  PCCursor()
	endIf
	return True
endIf

; Current (when distinct from focus) case.
if sOrigin == "current" then
	let hwnd = getCurrentWindow()
	if stringContains(what, "m") then
		let o = getObjectFromEvent(hwnd, -4, 0, childID)
		if !o then
			return False
		endIf
	elif stringContains(what, "n") then
		let childID = jcp___vcast(navGetCurrentObjectID())
	endIf
	return True
endIf

outputDebugString(formatString("jcp__setCustomProps: Unsupported origin: %1", sOrigin))
return False
endFunction

void function jcp__reset()
; Force the next jcp__setAll to reload, not use the cache.
let jcp___lastTick = 0
let jcp___intTick = 0
endFunction

/*
void function jcp__setCustomProps(variant byRef props, string sOrigin)
; Set any custom properties required per situation.
; Callers are expected to override this function.
; props is empty on entry, can be modified, and must be taken by reference.
; Properties understood by default (property name case insignificant):
; Properties used by both Braille and speech:
;	Name, type, state
;	containerName, containerType
;	value, position, dlgText
;	itemPos, itemCount (convenient way to compose position)
;	Hotkey (used in SayTutorialHelpHotKey)
;	subtypeCode (default type for speech and SayTutorialHelp and returned by BrailleCallbackObjectIdentify for this control)
;	attributes (CTRL_* constants from hjconst.jsh defining the control's states)
;	beforeField, afterField (text to speak/Braille before/after this field)
; Speech-only properties:
;	tutor, beforeTutor (tutor overrides the normal tutor message for the subtype)
; Braille-only properties:
;	dlgPageName, contextHelp, time, level
; Special properties internal to this system:
;	Mask:  Masks all special handling (use with caller name, e.g., jcp__set(props, "Mask.SayLine", True)).
;	Origin:  The origin to use to get default property values with getObject* functions (usually the sOrigin value used to create this property set)
; Note that SubtypeCode will set Type if Type is not set already; same for Attributes and State.
; sOrigin values that must be handled:
;	focus, current
;	subtype<subtype> (for BrailleAddObject* functions)
;	hwnd<handle>
; Other possible sOrigin values:
;	point<x>,<y>:  Like current but at a specific point
;	obj[<level>]  (level 0 is current)
;	hwnd<handle>/{nav<id>|sdm<id>|MSAA<id>}:  Nav* function ID, SDM ID, or MSAA ID within a window
;	point<x>,<y>:{hwnd|nav|SDM|MSAA}:  Like the extended hwnd syntax but from a specific point
endFunction
*/

variant function jcp___cachedVal(string which)
var int tc let tc = getTickCount()
if jcp___intTick && tc -jcp___intTick < 250 then
	if jcp___intcache.exists(which) then
		return jcp___intcache(which)
	endIf
else
	jcp___intcache.removeAll()
	let jcp___intTick = tc
endIf
; The sought value does not exist now.
if which == "stc" then
	jcp___intcache.add(which, getObjectSubtypeCode())
else
	; An id request.
	jcp___intcache.add(which, jcp__getID(which))
endIf
return jcp___intcache(which)
endFunction

int function jcp___sayControl(handle hwnd, string name, string type, string state, string containerName, string containerType, string value, string position, string dialogText)
var
	string allAttribs,
	int markupIncluded
let allAttribs = name +type +state +containerName +containerType +value +position +dialogText
if !allAttribs then
	return False
endIf
let markupIncluded = (stringContains(stringLower(allAttribs), "<voice") > 0)
if !markupIncluded then
	let name = SMMReplaceSymbolsWithMarkup(name)
	let type = SMMReplaceSymbolsWithMarkup(type)
	let state = SMMReplaceSymbolsWithMarkup(state)
	let containerName = SMMReplaceSymbolsWithMarkup(containerName)
	let containerType = SMMReplaceSymbolsWithMarkup(containerType)
	let value = SMMReplaceSymbolsWithMarkup(value)
	let position = SMMReplaceSymbolsWithMarkup(position)
	let dialogText = SMMReplaceSymbolsWithMarkup(dialogText)
endIf
if dialogText then
	sayControlExWithMarkup(hwnd, name, type, state, containerName, containerType, value, position, dialogText)
elif position then
	sayControlExWithMarkup(hwnd, name, type, state, containerName, containerType, value, position)
elif value then
	sayControlExWithMarkup(hwnd, name, type, state, containerName, containerType, value)
elif containerType then
	sayControlExWithMarkup(hwnd, name, type, state, containerName, containerType)
elif containerName then
	sayControlExWithMarkup(hwnd, name, type, state, containerName)
elif state then
	sayControlExWithMarkup(hwnd, name, type, state)
elif type then
	sayControlExWithMarkup(hwnd, name, type)
else
	sayControlExWithMarkup(hwnd, name)
endIf
return True
endFunction

variant function jcp___dispatch(string func)
return formatStringWithEmbeddedFunctions("<" +func +">")
endFunction

void function jcp__say(variant props, string caller)
var
	string origin,
	string callerName,
	string actionClass,
	string beforeField, string afterField,
	string name, string type, string state, string containerName, string containerType, string value, string position, string dialogText,
	int rowIdx, int rowCount, int colIdx, int colCount,
	int typeCode, int attributes,
	handle hwnd
let callerName = stringSegment(caller, "(", 1)
if jcp__get(props, "Mask."+callerName) then
	jcp___dispatch(caller)
	return
endIf
let origin = jcp__get(props, "origin")
if origin == "focus" then
	let hwnd = getFocus()
elif origin == "current" then
	let hwnd = getCurrentWindow()
elif origin == "hwnd" then
	let hwnd = stringToHandle(stringChopLeft(origin, 4))
else
	beep()
	jcp___dispatch(caller)
	return
endIf

let name = jcp__get(props, "name."+callerName)
let type = jcp__get(props, "type."+callerName)
if !type then
	let typeCode = jcp__get(props, "subtypeCode."+callerName)
	if typeCode then
		let type = smmStripMarkup(smmGetStartMarkupForControlType(typeCode))
		if !type then
			let type = " "
		endIf
	endIf
endIf
let state = jcp__get(props, "state."+callerName)
if !state then
	let attributes = jcp__get(props, "attributes."+callerName)
	if attributes then
		if !typeCode then
			let typeCode = 1  ; all seem to work equally well
		endIf
		let state = smmStripMarkup(smmGetStartMarkupForControlState(typeCode, attributes))
		if !type then
			let type = " "
		endIf
	endIf
endIf
let containerName = jcp__get(props, "containerName."+callerName)
let containerType = jcp__get(props, "containerType."+callerName)
let value = jcp__get(props, "value."+callerName)
let position = jcp__get(props, "position."+callerName)
let dialogText = jcp__get(props, "DlgText."+callerName)
; beforeField is useful when assigning a name adversely changes other elements of how JAWS says a control.
let beforeField = jcp__get(props, "beforeField."+callerName)
; afterField is useful for things like "minutes" in "Mark me away after _____ minutes"
let afterField = jcp__get(props, "afterField."+callerName)

if origin == "hwnd" then
	saveCursor() invisibleCursor() saveCursor()
	moveToWindow(hwnd)
endIf

let actionClass = jcp__getActionClass()
	if caller != "sayLine" && actionClass != "arrow" && beforeField then
		say(beforeField, OT_Control_Name, True)
	endIf

if caller == "sayLine" || actionClass == "arrow" then
	let containerName = ""
	let containerType = ""
	let dialogText = ""
endIf

if stringLength(name +type +state +containerName +containerType +value +position +dialogText) then
	; Use SayControlEx or SayControlExWithMarkup.
	var int customName, int customType, int customState, int customValue, int customPosition
	let customName = stringLength(name)
	if !name then let name = getObjectName() endIf
	let customType = stringLength(type)
	if !type then
		; Convert types that JAWS normally does not announce directly.
		var int tcode let tcode = getObjectSubtypeCode()
		var int tcode1 let tcode1 = 0
		if tcode == WT_ListItem && WT_ListItem == 86 then
			; WT_List == 85 but some non-English JAWS versions are
			; missing this constant in hjconst.jsh.
			let tcode1 = 85
		elif tcode == WT_ListBoxItem then
			let tcode1 = WT_ListBox
		elif tcode == WT_ListViewItem then
			let tcode1 = WT_ListView
		elif tcode == WT_TreeViewItem then
			let tcode1 = WT_TreeView
		endIf
		if tcode1 then
			let type = smmStripMarkup(smmGetStartMarkupForControlType(tcode1))
		else
			let type = getObjectSubtype()
		endIf
	endIf
	let customState = stringLength(state)
	if !state then
		let state = getObjectState()
		if stringCompare(stringStripAllBlanks(state), jcp___stateSelected, False) == 0 then
			; This normally doesn't happen, but JAWS doesn't say it normally anyway, so if it does show up, get rid of it.
			let state = " "
		endIf
	endIf
	; No function for containerName/Type.
	let customValue = stringLength(value)
	if !value then let value = getObjectValue() endIf
	let customPosition = stringLength(position)
	if !position then let position = positionInGroup() endIf
	; TODO: No default dialog text here, could try getDialogStaticText().

	; Caller-specific adjustments.
	if caller == "sayLine" || actionClass == "arrow" then
		let containerName = " "
		let dialogText = " "
		if !stringContains(":button:radiobutton:edit:readonlyedit:", ":"+stringLower(stringStripAllBlanks(type))+":") then
			let type = " "
		endIf
		if value then
			let name = " "
		endIf
	elif caller == "sayObjectActiveItem" then
		let beforeField = ""
		let type = " "
		let containerName = " "
		let dialogText = " "
		if stringContains(caller, "0") then
			; Position info is not to be spoken.
			let position = " "
		endIf
	endIf

	if !jcp___sayControl(hwnd, name, type, state, containerName, containerType, value, position, dialogText) then
		jcp___dispatch(caller)
	endIf
else
	jcp___dispatch(caller)
endIf  ; stringLength(name +type +state +containerName +containerType +value +position +dialogText)

let rowIdx = jcp__get(props, "row."+callerName)
if caller != "sayLine" && rowIdx then
	var string tblfmt
	let rowCount = jcp__get(props, "rows."+callerName)
	let colIdx = jcp__get(props, "col."+callerName)
	let colCount = jcp__get(props, "cols."+callerName)
	let tblfmt = "row %1"
	if rowCount then
		let tblfmt = tblfmt +" of %2"
	endIf
	if colIdx then
		let tblfmt = tblfmt +", column %3"
		if colCount then
			let tblfmt = tblfmt +" of %4"
		endIf
	endIf
	sayUsingVoice(VCTX_Message, formatString(tblfmt, intToString(rowIdx), intToString(rowCount), intToString(colIdx), intToString(colCount)), OT_Position)
endIf  ; rowIdx

if caller != "sayLine" && actionClass != "arrow" && afterField then
	say(afterField, OT_Control_Name, True)
endIf
if caller != "sayLine" && actionClass != "arrow" then
	; Tutor messages don't speak on SayLine, so beforeTutor doesn't either.
	var
		string beforeTutor
	let beforeTutor = jcp__get(props, "beforeTutor")
	if !stringIsBlank(beforeTutor) then
		; See JAWS 8's tutorialHelp.jss::sayTutorialHelp() for the rationale for using this SayUsingVoice call.
		; TODO: This may speak in a few undesirable places depending on user settings.
		sayUsingVoice(VCTX_Message, beforeTutor, OT_Line)
	endIf
endIf
endFunction

void function jcp___setCaller(int setting)
if setting then
	let jcp___key = getCurrentScriptKeyName()
	let jcp___caller = getScriptAssignedTo(jcp___key)
	; Clear that shortly.
	scheduleFunction("jcp___setCaller", 5)
else
	let jcp___key = ""
	let jcp___caller = ""
endIf
endFunction

void function keyPressedEvent(int nKey, string strKeyName, int nIsBrailleKey, int nIsScriptKey)
; Reset the cache on scripts in case they change focus, values, etc.
if nIsBrailleKey || nIsScriptKey then
	jcp__reset()
	jcp___setCaller(True)
else
	jcp___setCaller(False)
endIf
keyPressedEvent(nKey, strKeyName, nIsBrailleKey, nIsScriptKey)
endFunction

void function sayObjectTypeAndText(int nLevel)
; An override to improve handling of some windows.
var
	variant props,
	handle hwnd,
	string name,
	string buf
if jcp___disabled || jcp___inside || !isPCCursor() || nLevel > 0 then
	; System disabled, recursive call, non-PC cursor situation, or JAWS 8+ call on non-current control; handle normally.
	sayObjectTypeAndText(nLevel)
	return
endIf

; Look for special handling of the current control.
if jcp__setAll(props, "current", True) then
	let jcp___inside = True
	jcp__say(props, "sayObjectTypeAndText(" +intToString(nLevel) +")")
	let jcp___inside = False
	return
endIf

sayObjectTypeAndText(nLevel)
endFunction

void function sayWindowTypeAndText(handle hwnd)
; Look for special handling of the indicated window.
var
	variant props,
	string sOrigin
if jcp___disabled || jcp___inside then
	sayWindowTypeAndText(hwnd)
	return
endIf

if hwnd == getFocus() then
	let sOrigin = "focus"
elif hwnd == getCurrentWindow() then
	let sOrigin = "current"
else
	let sOrigin = "hwnd"+intToString(hwnd)
endIf
if !jcp___inside && jcp__setAll(props, sOrigin, True) then
	let jcp___inside = True
	jcp__say(props, "sayWindowTypeAndText(" +intToString(hwnd) +")")
	let jcp___inside = False
	return
endIf
sayWindowTypeAndText(hwnd)
endFunction

void function sayLine(int iDrawHighlights, int bSayingLineAfterMovement)
; Look for special handling of the current line.
var
	variant props,
	string sOrigin
if jcp___disabled || jcp___inside || !isPCCursor() then
	sayLine(iDrawHighlights, bSayingLineAfterMovement)
	return
endIf
let sOrigin = "current"
if !jcp__setAll(props, sOrigin, True) then
	sayLine(iDrawHighlights, bSayingLineAfterMovement)
	return
endIf
var int isMLE let isMLE = (
	getObjectSubtypeCode() == WT_MultiLine_Edit
; Catches controls that are being specifically set to this type.
	|| jcp__get(props, "SubtypeCode") == WT_MultiLine_Edit
; Catches controls that JAWS doesn't know are multiline edits.
; The second condition catches read-only multiline edits, which getObjectSubtypeCode() just calls WT_ReadOnlyEdit.
; the SendMessage checks for if the control accepts EM_SetSel, which implies that it's actually an edit control.
; 0x87 is WM_GetDlgCode, and 8 is DLGC_HasSetSel.
	|| ((getWindowStyleBits(getCurrentWindow()) & ES_Multiline)
		&& (sendMessage(getCurrentWindow(), 0x87, 0, 0) & 8))
)
if isMLE then
	; Avoid special processing on up/down arrows.
	jcp__set(props, "Mask.arrow", True)
	jcp__set(props, "Mask.sayLine", True)
	; Don't say a control name for SayLine in a multiline edit when it contains something.
	if getObjectValue()
	; or WM_GetTextLength returns non-zero.
	|| (sendMessage(getCurrentWindow(), 0x0E, 0, 0) > 0) then
		jcp__set(props, "Name", " ")
	endIf
endIf
let jcp___inside = True
jcp__say(props, "sayLine(" +intToString(iDrawHighlights) +", " +intToString(bSayingLineAfterMovement) +")")
let jcp___inside = False
endFunction

void function sayObjectActiveItem(int sayPositionInfo)
; Look for special handling of the current item.
var
	variant props,
	string sOrigin
if jcp___disabled || jcp___inside then
	sayObjectActiveItem(sayPositionInfo)
	return
elif getObjectSubtypeCode() == WT_MultiLine_Edit
; The second condition catches read-only multiline edits, which getObjectSubtypeCode() just calls WT_ReadOnlyEdit.
; the SendMessage checks for if the control accepts EM_SetSel, which implies that it's actually an edit control.
; 0x87 is WM_GetDlgCode, and 8 is DLGC_HasSetSel.
|| ((getWindowStyleBits(getCurrentWindow()) & ES_Multiline) && (sendMessage(getCurrentWindow(), 0x87, 0, 0) & 8)) then
	sayObjectActiveItem(sayPositionInfo)
	return
endIf

let sOrigin = "current"
if isPCCursor() && jcp__setAll(props, sOrigin, True) then
	let jcp___inside = True
	jcp__say(props, "sayObjectActiveItem(" +intToString(sayPositionInfo) +")")
	let jcp___inside = False
	return
endIf
sayObjectActiveItem(sayPositionInfo)
endFunction

int function getTreeViewLevel()
if jcp___disabled || jcp___inside || !isPCCursor() then
	return getTreeViewLevel()
endIf
var variant props
if jcp__setAll(props, "current", False) then
	if props.exists("level") then
		return stringToInt(jcp__get(props, "Level"))
	endIf
endIf
return getTreeViewLevel()
endFunction

int function sayTutorialHelp(int iSubType, int isScriptKey)
; Blocks tutor messages when set to blank.
; Also allows iSubtype to be overridden by a subtypeCode property in props.
; Finally allows the subtype to be obtained from a different origin (a specific window).
var
	variant props,
	string msg
if jcp___disabled || jcp___inside || !isPCCursor() || getObjectSubtypeCode() != iSubtype then
	; We know nothing of this, so let JAWS figure it out.
	return sayTutorialHelp(iSubType, isScriptKey)
endIf
if jcp__setAll(props, "current", False) then
	let msg = jcp__get(props, "tutor")
	if stringLength(msg) > 0 && stringIsBlank(msg) then
		; Somebody said say nothing, so say nothing.
		return True  ; but say we said what we needed to say.
	elif jcp__getActionClass() == "arrow" then
		; No tutor text on arrows.
		return True
	endIf
	var int iSubtype1
	let iSubtype1 = jcp__get(props, "subtypeCode")
	if iSubtype1 then
		let iSubtype = iSubtype1
	else
		; This is just for when jcp__setCustomProps() changes the origin.
		var string sOrigin, handle hwnd
		let sOrigin = jcp__get(props, "origin")
		if sOrigin == "hwnd" then
			let hwnd = stringToHandle(stringChopLeft(sOrigin, 4))
			saveCursor() invisibleCursor() saveCursor()
			moveToWindow(hwnd)
			let iSubtype = getObjectSubtypeCode()
		endIf  ; sOrigin == "hwnd"
	endIf  ; iSubtype1, else
endIf  ; jcp__setAll got something
return sayTutorialHelp(iSubType, isScriptKey)
endFunction

int function sayTutorialHelpHotKey(handle hwnd, int isScriptKey)
var
	variant props,
	string msg
if jcp___disabled || jcp___inside || !isPCCursor() || hwnd != getFocus() then
	; We know nothing of this, so let JAWS figure it out.
	return sayTutorialHelpHotKey(hwnd, isScriptKey)
endIf
if jcp__setAll(props, "current", False) then
	let msg = jcp__get(props, "hotkey")
	if stringLength(msg) > 0 then
		if stringIsBlank(msg) then
			; Somebody said say nothing, so say nothing.
		elif jcp__getActionClass() == "arrow" then
			; No tutor text on arrows.
		else
			sayUsingVoice(VCTX_Message, msg, OT_Access_Key)
		endIf
		return True
	endIf
	; This is just for when jcp__setCustomProps() changes the origin.
	var string sOrigin
	let sOrigin = jcp__get(props, "origin")
	if sOrigin == "hwnd" then
		let hwnd = stringToHandle(stringChopLeft(sOrigin, 4))
		saveCursor() invisibleCursor() saveCursor()
		moveToWindow(hwnd)
	endIf
endIf
sayTutorialHelpHotKey(hwnd, isScriptKey)
endFunction

string function getCustomTutorMessage()
; Called in JAWS 7.1+ to get custom tutor messages.
var
	variant props,
	string msg
let msg = ""
if jcp___disabled || jcp___inside || !jcp__setAll(props, "current", False) then
	return getCustomTutorMessage()
endIf
let msg = jcp__get(props, "tutor")
if msg then
	return msg
endIf
return getCustomTutorMessage()
endFunction

int function jcp___BrailleAddObjectHelper0(string whichProp, int nSubtype)
; Logic for BrailleAddObject* functions to use (wrapped by jcp___BrailleAddObjectHelper though).
var
	string sOrigin,
	variant props,
	int attribs,
	int x, int y,
	string val
if jcp___disabled then
	return False
endIf
var int scode let scode = jcp___cachedVal("stc")
if scode && scode != nSubtype then
	; BrailleAddObject* function(s) called on a parent dialog control.
	var int level, int found
	let level = 1
	let found = False
	while !found && level <= 5
		if getObjectSubTypeCode(False, level) == nSubtype then
			let sOrigin = "focus" +intToString(level)
			let found = True
		endIf
		let level = level +1
	endWhile
	if !found then
		; TODO:  This may cause some incorrect behavior, but it is hard to know when this should or shouldn't be done.
		; Example:  A .Net edit combo/spin box has been seen to return Spinbox for Braille and Edit Combo for speech.
		; GetObjectSubtypeCode returns ReadOnly Edit at level 0 and Combobox at level 1.
		;let sOrigin = "subtype" +intToString(nSubtype)
		let sOrigin = "focus"
	endIf
else
	let sOrigin = "focus"
endIf
if !jcp__setAll(props, sOrigin, False) then
	return False
endIf
if whichProp == "name" then
	let val = jcp__get(props, "row.brl")
	if val then
		var variant val1
		let val1 = jcp__get(props, "col.brl")
		if val1 then
			let val = formatString("r%1c%2", val, val1)
		else
			let val = formatString("r%1", val)
		endIf
		BrailleAddString(val, 0,0, 0)
	endIf
	let val = jcp__get(props, "beforeField.brl")
	if val then
		BrailleAddString(val, 0,0, 0)
	endIf
endIf
let val = jcp__get(props, whichProp +".brl")
if !val then
	if whichProp == "type" then
		let scode = jcp__get(props, "subtypeCode.brl")
		if scode then
			let val = BrailleGetSubtypeString(scode)
			if !val then
				let val = " "
			endIf
		endIf
	elif whichProp == "state" then
		let attribs =jcp__get(props, "attributes")
		if attribs then
			let val = BrailleGetStateString (attribs)
			if !val then
				let val = " "
			endIf
		endIf
	endIf
endIf
var int retval let retval = False
if val then
	if !stringIsBlank(val) then
		let x = 0
		let y = 0
		; "name" left out deliberately; prevents cursor from being placed on label instead of value in edit fields
		if whichProp == "type" || whichProp == "state" || whichProp == "value" then
			let x = getCursorCol()
			let y = getCursorRow()
		endIf
		;if whichProp == "type" then sayString(val) endIf
		BrailleAddString(val, x, y, 0)
	endIf  ; !stringIsBlank(val)
	let retval = True
endIf  ; val
if whichProp == "value" then
	let val = jcp__get(props, "afterField.brl")
	if val then
		if !retval then
			; No custom value, but put any non-custom one here.
			; The BrailleAddObjectValue in this file should be our caller and so shouldn't run again.
			BrailleAddObjectValue(nSubtype)
		endIf
		BrailleAddString(val, 0,0, 0)
	endIf
endIf
return retval
endFunction

int function jcp___BrailleAddObjectHelper(string whichProp, int nSubtype)
; Called by all BrailleAddObject* functions.
var
	int retval
let retval = jcp___BrailleAddObjectHelper0(whichProp, nSubtype)
if retval then
	return retval
endIf
return jcp___dispatch("BrailleAddObject" +whichProp +"(" +intToString(nSubtype) +")")
endFunction

; These are all the BrailleAddObject* functions internally recognized as of JAWS 8.0.
int function BrailleAddObjectName(int nSubtype)
return jcp___BrailleAddObjectHelper("Name", nSubtype)
endFunction
int function BrailleAddObjectType(int nSubtype)
return jcp___BrailleAddObjectHelper("Type", nSubtype)
endFunction
int function BrailleAddObjectState(int nSubtype)
return jcp___BrailleAddObjectHelper("State", nSubtype)
endFunction
int function BrailleAddObjectValue(int nSubtype)
return jcp___BrailleAddObjectHelper("Value", nSubtype)
endFunction
int function BrailleAddObjectContainerName(int nSubtype)
return jcp___BrailleAddObjectHelper("ContainerName", nSubtype)
endFunction
int function BrailleAddObjectContainerType(int nSubtype)
return jcp___BrailleAddObjectHelper("ContainerType", nSubtype)
endFunction
int function BrailleAddObjectPosition(int nSubtype)
return jcp___BrailleAddObjectHelper("Position", nSubtype)
endFunction
int function BrailleAddObjectDlgPageName(int nSubtype)
return jcp___BrailleAddObjectHelper("DlgPageName", nSubtype)
endFunction
int function BrailleAddObjectDlgText(int nSubtype)
return jcp___BrailleAddObjectHelper("DlgText", nSubtype)
endFunction
int function BrailleAddObjectContextHelp(int nSubtype)
return jcp___BrailleAddObjectHelper("ContextHelp", nSubtype)
endFunction
int function BrailleAddObjectTime(int nSubtype)
return jcp___BrailleAddObjectHelper("Time", nSubtype)
endFunction
int function BrailleAddObjectLevel(int nSubtype)
return jcp___BrailleAddObjectHelper("Level", nSubtype)
endFunction

int function BrailleCallbackObjectIdentify()
; Allows WT_Unknown to be translated to jcp__get(props, "subtypeCode.brl").
; Set subtypeCode.brl to adjust the Brailled type and the set of BrailleAddObject* functions called for a control.
; Also allows the subtype to be obtained from a different origin (a specific window).
var int tc let tc = getTickCount()
var
	variant props,
	int scode
if !jcp___disabled && !jcp___inside && jcp__setAll(props, "current", False) then
	let scode = jcp__get(props, "subtypeCode.brl")
	if scode then
		return scode
	else
		; This is just for when jcp__setCustomProps() changes the origin.
		var string sOrigin, handle hwnd
		let sOrigin = jcp__get(props, "origin")
		if sOrigin == "hwnd" then
			let hwnd = stringToHandle(stringChopLeft(sOrigin, 4))
			saveCursor() invisibleCursor() saveCursor()
			moveToWindow(hwnd)
			let scode = getObjectSubtypeCode()
			return scode
		endIf  ; sOrigin == "hwnd"
	endIf
endIf
return BrailleCallbackObjectIdentify()
endFunction

int function jcp__set(object byRef props, string whichProp, variant val, /*?*/ int allowFunctionCall)
; Set a control property in props.
; Case is not significant in property names.
; If allowFunctionCall is False (default) and val begins with "@",
; a space is prepended to val to keep jcp__get() from thinking it's a function call.
let whichProp = stringLower(whichProp)
; Special itemPos/Count for setting parts of Position conveniently.
if whichProp == "itemPos" || whichProp == "itemCount" then
	var string posdata, int pos
	let posdata = jcp__get(props, "Position")
	if !posdata then
		let posdata = positionInGroup()
		if !posdata then
			let posdata = jcp___MOfN
		endIf
	endIf
	if whichProp == "itemPos" then
		let pos = stringContains(posdata, " ")
		let posdata = stringChopLeft(" " +intToString(val) +stringChopLeft(posdata, pos-1), 1)
	else
		let pos = stringContainsFromRight(posdata, " ")
		let posdata = stringChopLeft(" " +stringLeft(posdata, pos) +intToString(val), 1)
	endIf
	let whichProp = "position"
	let val = jcp___vcast(posdata)
endIf
if !allowFunctionCall && val == "@" then
	let val = jcp___vcast(" " +val)
endIf
props.remove(whichProp)
props.add(whichProp, val)
return True
endFunction

;======================== End module jcp (from jcpdict) ========================

